const { getJson } = require("serpapi");
const puppeteer = require("puppeteer");
const axios = require("axios");
const cheerio = require("cheerio");
const fs = require("fs").promises;
const schedule = require("node-schedule");
const pLimit = async () => {
  const module = await import('p-limit');
  return module.default || module;
};
const winston = require("winston");
const nodemailer = require("nodemailer");

// Logger configuration
const logger = winston.createLogger({
  level: "info",
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.json()
  ),
  transports: [
    new winston.transports.File({ filename: "error.log", level: "error" }),
    new winston.transports.File({ filename: "combined.log" }),
    new winston.transports.Console(),
  ],
});

// Email configuration
const transporter = nodemailer.createTransport({
  service: "ssl0.ovh.net",
  auth: {
    user: process.env.EMAIL_USER || "hello@job-finder-bot.com",
    pass: process.env.EMAIL_PASS || "JftV2q5jxB0FxjA8Sis",
  },
});

// Function to send email notifications
async function sendEmail(subject, body) {
  const mailOptions = {
    from: process.env.EMAIL_USER || "hello@job-finder-bot.com",
    to: "smessan@gmail.com",
    subject: subject,
    html: body,
  };

  try {
    await transporter.sendMail(mailOptions);
    logger.info(`Email sent: ${subject}`);
  } catch (error) {
    logger.error("Error sending email:", { error: error.message });
  }
}

//Fonction d'enregistrement des logs

async function sendLogs(pays, type, message) {
  try {
    const response = await fetch('https://job-finder-bot.com/home/ajout_logs', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        pays,
        type,
        message,
      }),
    });

    const data = await response.json();
    console.log('✅ Réponse du serveur :', data);
  } catch (error) {
    console.error('❌ Erreur lors de l’envoi des logs :', error.message);
  }
}


// Configuration constants
const API_KEY = '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'; // Replace with your actual API key
const API_URL = 'https://openrouter.ai/api/v1/chat/completions';
const CONCURRENCY_LIMIT = 2;
const REQUEST_INTERVAL = 5000;
const MAX_RETRIES = 3;
const TIMEOUT = 60000;


async function initializeBrowser() {
  try {
    // Lancement du navigateur de façon plus légère
    const browser = await puppeteer.launch({
      headless: "new",
      args: [
        "--no-sandbox",
        "--disable-setuid-sandbox",
        "--disable-blink-features=AutomationControlled",
        "--disable-infobars",
        "--window-size=1280,720",
        "--disable-gpu",
        "--disable-dev-shm-usage", // réduit la mémoire partagée, crucial sur petit VPS
        "--single-process",         // moins de processus Chromium → moins de RAM
        "--no-zygote",
      ],
      executablePath: "/home/ubuntu/.cache/puppeteer/chrome/linux-136.0.7103.94/chrome-linux64/chrome",
    });

    // Créer une page
    const page = await browser.newPage();

    // User agent et langue
    await page.setUserAgent(
      "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
    );
    await page.setExtraHTTPHeaders({ "Accept-Language": "fr-FR,fr;q=0.9" });

    // Masquer que c’est un navigateur automatisé
    await page.evaluateOnNewDocument(() => {
      Object.defineProperty(navigator, "webdriver", { get: () => false });
    });

    // Timeout plus long pour éviter les crashs sur pages lentes
    page.setDefaultNavigationTimeout(60000); // 60s
    page.setDefaultTimeout(60000);           // 60s pour autres actions

    return { browser, page };
  } catch (error) {
    logger.error("Error initializing browser:", { error: error.message });
    throw error;
  }
}

// Generic scrape function with retry logic
async function scrapeWithRetry(page, url, selector, extractFn, maxRetries = MAX_RETRIES) {
  let retries = 0;
  while (retries < maxRetries) {
    try {
      logger.info(`Loading ${url}...`);
      await page.goto(url, { waitUntil: "domcontentloaded", timeout: TIMEOUT });
      await page.waitForSelector(selector, { timeout: 15000, visible: true });
      const data = await page.evaluate(extractFn);
      logger.info(`Retrieved ${data.length} items from ${url}`);
      return data;
    } catch (error) {
      retries++;
      logger.warn(`Attempt ${retries} failed for ${url}: ${error.message}`);
      if (retries === maxRetries) {
        logger.error(`Failed to scrape ${url} after ${maxRetries} retries`);
        return [];
      }
      await new Promise((r) => setTimeout(r, 2000 * Math.pow(2, retries))); // Exponential backoff
    }
  }
  return [];
}



// Scrape emploi.cd
async function scrapeEmploiRDC(page) {
  const jobs = [];
  for (let pageNum = 1; pageNum <= 5; pageNum++) {
    const url = `https://www.emploi.cd/recherche-jobs-congo-rdc?page=${pageNum}`;
    const pageJobs = await scrapeWithRetry(
      page,
      url,
      "div.card.card-job",
      () => {
        const jobList = [];
        document.querySelectorAll("div.card.card-job").forEach((job) => {
          // Titre du poste
          const title = job.querySelector("h3 a")?.innerText.trim() || "Non spécifié";

          // Nom de l’entreprise (sinon Confidentiel)
          const company = job.querySelector("picture img")?.getAttribute("title")?.trim() || "Confidentiel";

          // Lien vers l’offre
          const link = job.dataset.href || "#";

          jobList.push({
            title,
            source: "Emploi RDC",
            link
          });
        });
        return jobList;
      }
    );
    jobs.push(...pageJobs);
    await new Promise((r) => setTimeout(r, REQUEST_INTERVAL)); // pause anti-ban
  }
  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

//Scrape Relief web

async function scrapeReliefWebRDC(page) {
  const url =
    "https://reliefweb.int/jobs?advanced-search=%28C75%29&list=Democratic+Republic+of+the+Congo+%28DR+Congo%29+Jobs";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "article.rw-river-article--job",
    () => {
      const jobList = [];

      document.querySelectorAll("article.rw-river-article--job").forEach((job) => {
        // Titre et lien
        const titleEl = job.querySelector("h3.rw-river-article__title a");
        const title = titleEl?.innerText.trim() || "Non spécifié";
        const link = titleEl?.href || "#";

        // Organisation
        const companyEl = job.querySelector(
          "dd.rw-entity-meta__tag-value--source a"
        );
        const company = companyEl?.innerText.trim() || "Confidentiel";

        // Date de publication
        const postedEl = job.querySelector(
          "dd.rw-entity-meta__tag-value--posted time"
        );
        const posted = postedEl?.getAttribute("datetime") || "";

        // Date de clôture
        const closingEl = job.querySelector(
          "dd.rw-entity-meta__tag-value--closing-date time"
        );
        const closing = closingEl?.getAttribute("datetime") || "";

        jobList.push({
          title,
          source: "ReliefWeb RDC",
          link
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}


//scrape UNJobsRDC
async function scrapeUnjobsRDC(page) {
  return scrapeWithRetry(
    page,
    "https://unjobs.org/duty_stations/democratic-republic-of-congo",
    ".job", // Primary selector, fallback below if needed
    async () => {
      // Wait for the page to stabilize (network idle or specific selector)
      try {
        await page.waitForSelector(".job", { timeout: 20000 }); // Increased timeout to 20s
      } catch (e) {
        console.warn("Primary selector '.job' not found, trying fallback selector");
        await page.waitForSelector("div[id*='job']", { timeout: 20000 }); // Fallback to divs with 'job' in ID
      }

      const jobList = await page.evaluate(() => {
        const jobs = [];
        // Try primary selector, fallback to broader selector if needed
        const jobElements = document.querySelectorAll(".job")?.length
          ? document.querySelectorAll(".job")
          : document.querySelectorAll("div[id*='job']");

        jobElements.forEach((job) => {
          const titleElement = job.querySelector(".jtitle") || job.querySelector("a[href*='vacancies']");
          const title = titleElement?.textContent.trim() || "";
          const link = titleElement?.getAttribute("href") || "#";
          
          // Extract company (next sibling after <a> or fallback)
          const company = job.childNodes[2]?.textContent.trim() || 
                         job.querySelector("*:nth-child(2)")?.textContent.trim() || "Unknown";
          
          // Extract date
          const date = job.querySelector(".upd.timeago")?.getAttribute("datetime") || 
                       job.querySelector("time")?.getAttribute("datetime") || "";
          
          // Extract location from title or fallback to "Benin"
          const location = title.match(/, (.+)$/)?.[1] || "Benin";

          if (title && link !== "#") {
            jobs.push({
              title,
              source: "UNjobs",
              link
            });
          }
        });
        return jobs;
      });

      return jobList;
    },
    { retries: 3, timeout: 30000 } // Increase retries and timeout for robustness
  );
}

//Scraping Option Carriere RDC
async function scrapeOptionCarriereRDC(page) {
  const url = "https://www.optioncarriere.cd/emploi/RD-Congo";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "article.job.clicky",
    () => {
      const jobList = [];

      document.querySelectorAll("article.job.clicky").forEach((job) => {
        // ----- Titre & lien -----
        const titleEl = job.querySelector("h2 a");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        // data-url est un chemin relatif -> on le complète avec le domaine
        const relativeLink = job.getAttribute("data-url") || titleEl?.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.optioncarriere.cd${relativeLink}`;

        // ----- Entreprise -----
        const companyEl = job.querySelector("p.company a");
        const company = companyEl?.innerText.trim() || "Confidentiel";

        // ----- Localisation -----
        // Le site affiche plusieurs <li> : pays + ville (ex : Congo / Kinshasa)
        const locationEls = Array.from(job.querySelectorAll("ul.location li"));
        const location = locationEls.map(li => li.innerText.trim()).join(" - ");

        // ----- Date / temps écoulé -----
        // Ex: "Il y a 22 heures"
        const postedEl = job.querySelector("footer ul.tags li span.badge");
        const posted = postedEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "OptionCarriere RDC",
          link
          
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

//Scraping Ton Job

async function scrapeTonJob(page) {
  const url = "https://tonjob.net/";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "div.job-list.post-2001, div.job-list", // sélectionne les conteneurs d’offres
    () => {
      const jobList = [];

      document.querySelectorAll("div.job-list").forEach((job) => {
        // ----- Titre -----
        const titleEl = job.querySelector("h3.title-job-list a");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        // Filtrer les offres expirées
        if (/Attention\s*:\s*Délai de candidature dépassé/i.test(title)) {
          return; // on ignore cette offre
        }

        // ----- Lien -----
        const link = titleEl?.href || "#";

        // ----- Entreprise -----
        const companyEl = job.querySelector(".company strong.text-theme");
        const company = companyEl?.innerText.trim() || "Confidentiel";

        // ----- Localisation -----
        const locationEl = job.querySelector(".job-metas .location");
        const location = locationEl?.innerText.trim() || "";

        // ----- Type de contrat (CDD, CDI, etc.) -----
        const typeEl = job.querySelector(".job-type");
        const contractType = typeEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "TonJob",
          link
          
          
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}


// Scraping Impactpool en évitant les offres fermées
async function scrapeImpactPoolRDC(page) {
  const url = "https://www.impactpool.org/countries/Congo Democratic";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "div.job",
    () => {
      const jobList = [];

      document.querySelectorAll("div.job").forEach((job) => {
        const linkEl = job.querySelector("a");
        if (!linkEl) return;

        // ----- Vérifier si l'offre est fermée -----
        const closedBadge = job.querySelector(".ip-badge-text");
        if (closedBadge && /Job is closed/i.test(closedBadge.textContent)) {
          return; // on saute cette offre
        }

        // ----- Lien de l'offre -----
        const relativeLink = linkEl.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.impactpool.org${relativeLink}`;

        // ----- Titre -----
        const titleEl = linkEl.querySelector("div[type='cardTitle']");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        // ----- Entreprise / Organisation -----
        const companyEl = linkEl.querySelector("div[type='bodyEmphasis']");
        const company = companyEl?.innerText.trim() || "Confidentiel";

        // ----- Localisation -----
        const locationEl = linkEl.querySelector(
          "div[type='bodyEmphasis'][style*='color: #63625B']"
        );
        const location = locationEl?.innerText.trim() || "";

        // ----- Niveau / type de poste -----
        const levelEl = linkEl.querySelector(
          "div[type='bodyEmphasis'][style*='color: #8A8881']"
        );
        const level = levelEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "ImpactPool RDC",
          link
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

// Scraping  Tala-com.
async function scrapeTalaComRDC(page) {
  const url = "https://www.tala-com.com/travailler-rdc/offre-emploi";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "div.views-row.job-posting-item",
    () => {
      const jobList = [];

      document.querySelectorAll("div.views-row.job-posting-item").forEach((job) => {
        // ----- Titre & lien -----
        const titleEl = job.querySelector(".views-field-title a");
        const title = titleEl?.innerText.trim() || "Non spécifié";
        const relativeLink = titleEl?.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.tala-com.com${relativeLink}`;

        // ----- Date de publication -----
        const dateEl = job.querySelector(".views-field-created .field-content");
        const published = dateEl?.innerText.trim() || "";

        // ----- Domaine d'activité -----
        const domainEl = job.querySelector(
          ".views-field-field-domaine-activite .field-content"
        );
        const domain = domainEl?.innerText.trim() || "";

        // ----- Ville -----
        const cityEl = job.querySelector(".views-field-field-ville .field-content");
        const city = cityEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "Tala-com RDC",
          link
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

// Scraping du site Devex
async function scrapeDevexRDC(page) {
  const url =
    "https://www.devex.com/jobs/search?filter%5Blocations%5D%5B%5D=Congo%2C+The+Democratic+Republic+of&sorting%5Border%5D=desc";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "div._jobItem_9z2cy_7.position-relative", // Sélecteur parent d'une offre
    () => {
      const jobList = [];

      document.querySelectorAll("div._jobItem_9z2cy_7.position-relative").forEach((job) => {
        // ---- Titre & lien ----
        const titleEl = job.querySelector('[data-testid="job-details-page-link"] a');
        const title = titleEl?.innerText.trim() || "Non spécifié";

        const relativeLink = titleEl?.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.devex.com${relativeLink}`;

        // ---- Société ----
        const companyEl = job.querySelector(".secondary-info span");
        const company = companyEl?.innerText.trim() || "Confidentiel";

        // ---- Lieu ----
        const placeEl = job.querySelector('[data-testid="job-places-value"]');
        const location = placeEl?.innerText.trim() || "";

        // ---- Date de publication ----
        const publishedEl = job.querySelector('[data-testid="published-date-section"]');
        const published = publishedEl?.innerText.trim() || "";

        // ---- Date de clôture ----
        const closingEl = job.querySelector('[data-testid="closing-date"] span.ms-1');
        const closing = closingEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "Devex RDC",
          link,
         
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

//Scraping Michael Page Africa
async function scrapeMichaelPageRDC(page) {
  const url =
    "https://www.michaelpageafrica.com/fr/jobs/r%C3%A9publique-d%C3%A9mocratique-du-congo";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "li.views-row", // Chaque offre est dans un <li class="views-row">
    () => {
      const jobList = [];

      document.querySelectorAll("li.views-row").forEach((job) => {
        // ----- Titre & lien -----
        const titleEl = job.querySelector(".job-title h3 a");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        const relativeLink = titleEl?.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.michaelpageafrica.com${relativeLink}`;

        // ----- Localisation -----
        const locationEl = job.querySelector(".job-location");
        const location = locationEl?.innerText
          .replace(/\s*\n\s*/g, " ")
          .trim() || "";

        // ----- Type de contrat -----
        const contractEl = job.querySelector(".job-contract-type");
        const contract = contractEl?.innerText.trim() || "";

        // ----- Résumé -----
        const summaryEl = job.querySelector(".job_advert__job-summary-text");
        const summary = summaryEl?.innerText.trim() || "";

        // ----- Points clés (bullet points) -----
        const bullets = Array.from(
          job.querySelectorAll(".job_advert__job-desc-bullet-points li")
        ).map((li) => li.innerText.trim());

        jobList.push({
          title,
          source: "Michael Page Africa RDC",
          link
          
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

//Extraction Radio Kapi
async function scrapeRadioOkapiRDC(page) {
  const url = "https://www.radiookapi.net/offre-demploi";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "h2.field-content", // Chaque offre est dans un <h2 class="field-content">
    () => {
      const jobList = [];

      document.querySelectorAll("h2.field-content").forEach((job) => {
        // ----- Titre & lien -----
        const linkEl = job.querySelector("a");
        const title = linkEl?.innerText.trim() || "Non spécifié";

        const relativeLink = linkEl?.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.radiookapi.net${relativeLink}`;

        // Comme le site ne fournit pas d'autres infos (lieu, contrat, résumé…),
        // on se limite au titre et au lien.
        jobList.push({
          title,
          source: "Radio Okapi RDC",
          link
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

//Scrap developpement Aid 

// Scraping DevelopmentAid – RDC
async function scrapeDevelopmentAidRDC(page) {
  const url = "https://www.developmentaid.org/jobs/search?sort=highlighted.desc,postedDate.desc&locations=23";

  const jobs = await scrapeWithRetry(
    page,
    url,
    "da-search-card[entitytype='job']", // Chaque offre est dans ce composant
    () => {
      const jobList = [];

      document.querySelectorAll("da-search-card[entitytype='job']").forEach((job) => {
        // ----- Titre & lien -----
        const titleEl = job.querySelector(".search-card__title");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        const relativeLink = titleEl?.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.developmentaid.org${relativeLink}`;

        // ----- Organisation -----
        const orgEl = job.querySelector(".details-container a.search-card__link");
        const organization = orgEl?.innerText.trim() || "Confidentiel";

        // ----- Date de publication -----
        const postedEl = job.querySelector(".search-card__posted-detail span:nth-child(2)");
        const postedDate = postedEl?.innerText.trim() || "";

        // ----- Lieu -----
        const locationEl = job.querySelector(".search-card__additional-details span.ng-star-inserted:nth-child(2)");
        // Le sélecteur ci-dessus peut varier, on sécurise avec textContent
        const location = locationEl?.innerText.trim() || "";

        // ----- Type de contrat -----
        const contractEl = job.querySelector(".search-card__hidden-detail span:nth-child(2)");
        const contract = contractEl?.innerText.trim() || "";

        // ----- Expérience -----
        const expEl = job.querySelector(".search-card__hidden-detail.d-flex.ng-star-inserted span:nth-child(2)");
        const experience = expEl?.innerText.trim() || "";

        // ----- Deadline -----
        const deadlineEl = job.querySelector(".search-card__detail-for_lg + span");
        const deadline = deadlineEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "DevelopmentAid RDC",
          link
        });
      });

      return jobList;
    }
  );

  const filteredJobs = [];
  for (const job of jobs) {
    const nb = await jobExistv2(job.link); // appelle la fonction fournie
    if (nb === 0) {
      filteredJobs.push(job);
    }
  }

  return filteredJobs;
}

//Offres mon congo
// Scraping MonCongo Jobs avec filtrage jobExist
async function scrapeMonCongoJobs(page) {
  const url = "https://www.moncongo.com/jobs";

  const rawJobs = await scrapeWithRetry(
    page,
    url,
    "div.col",
    () => {
      const jobList = [];

      document.querySelectorAll("div.col").forEach((job) => {
        const card = job.querySelector(".cardStyle .card");
        if (!card) return;

        // ----- Titre -----
        const titleEl = card.querySelector("h3");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        // ----- Résumé -----
        const summaryEl = card.querySelector(".line-show-3");
        const summary = summaryEl?.innerText.trim() || "";

        // ----- Lien -----
        // Vérifie si un <a> est présent dans la carte
        let linkEl = card.querySelector("a");
        let link = linkEl?.getAttribute("href");
        if (!link) {
          // fallback : on ne peut pas construire le lien exact sans info, mettre #
          link = "#";
        } else if (!link.startsWith("http")) {
          link = `https://www.moncongo.com${link}`;
        }

        // ----- Image -----
        const imgEl = card.querySelector(".imgList img");
        const image = imgEl?.getAttribute("src") || "";

        jobList.push({
          title,
          summary,
          link,
          image,
          source: "MonCongo Jobs"
        });
      });

      return jobList;
    }
  );


  // ----- Filtrage avec jobExist -----
  const filteredJobs = [];
  for (const job of rawJobs) {
    if (job.link !== "#") {
      const nb = await jobExistv2(job.link);
      if (nb === 0) filteredJobs.push(job);
    }
  }

  return filteredJobs;
}


// Scraping UN Job Net – RDC avec filtrage jobExist
async function scrapeUNJobNetRDC(page) {
  const url = "https://www.unjobnet.org/countries/Democratic Republic of the Congo";

  // 1️⃣ Récupération brute des offres
  const rawJobs = await scrapeWithRetry(
    page,
    url,
    "div.row.border.border-bottom-0.bg-white.p-3.job",
    () => {
      const jobList = [];

      document.querySelectorAll("div.row.border.border-bottom-0.bg-white.p-3.job").forEach((job) => {
        // ----- Titre & lien -----
        const titleEl = job.querySelector("a.py-2.link-primary.h6.fw-bold");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        const relativeLink = titleEl?.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.unjobnet.org${relativeLink}`;

        // ----- Organisation -----
        const orgEl = job.querySelector("div.mt-2 a.link-dark");
        const organization = orgEl?.innerText.trim() || "Confidentiel";

        // ----- Localisation -----
        const locationEl = job.querySelector("div.mt-1 a.link-darkx");
        const location = locationEl?.innerText.trim() || "";

        // ----- Type de contrat -----
        const contractEl = job.querySelector("div.mt-1 span a[href*='apptypes']");
        const contract = contractEl?.innerText.trim() || "";

        // ----- Date de publication -----
        const postedEl = job.querySelector("div.mt-1 span small.text-muted");
        const postedDate = postedEl?.innerText.replace("Posted", "").trim() || "";

        jobList.push({
          title,
          source: "UN Job Net RDC",
          link
        });
      });

      return jobList;
    }
  );

  // 2️⃣ Filtrage asynchrone avec jobExist
  const filteredJobs = [];
  for (const job of rawJobs) {
    const nb = await jobExistv2(job.link); // appelle jobExist pour chaque lien
    if (nb === 0) {
      filteredJobs.push(job);
    }
  }

  // 3️⃣ Retourne uniquement les offres validées
  return filteredJobs;
}


// Scraping Bensizwe Jobs avec filtrage jobExist
async function scrapeBensizweJobs(page) {
  const url = "https://www.bensizwe.com/jobs";

  // 1️⃣ Récupération brute des offres
  const rawJobs = await scrapeWithRetry(
    page,
    url,
    "div.card.bg-white.h-100.shadow",
    () => {
      const jobList = [];

      document.querySelectorAll("div.card.bg-white.h-100.shadow").forEach((job) => {
        const linkEl = job.querySelector("a.text-decoration-none");
        if (!linkEl) return;

        // ----- Lien -----
        const relativeLink = linkEl.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://www.bensizwe.com${relativeLink}`;

        // ----- Titre -----
        const titleEl = job.querySelector("h5.card-title strong u");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        // ----- Type de contrat -----
        const contractEl = job.querySelector("span.s_badge");
        const contract = contractEl?.innerText.trim() || "";

        // ----- Localisation -----
        const locationEl = job.querySelector("i.fa-map-marker + span");
        const location = locationEl?.innerText.trim() || "";

        // ----- Deadline / date limite -----
        const deadlineEl = job.querySelector("i.fa-clock-o + span + span");
        const deadline = deadlineEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "Bensizwe Jobs",
          link,
          
        });
      });

      return jobList;
    }
  );

  // 2️⃣ Filtrage asynchrone avec jobExist
  const filteredJobs = [];
  for (const job of rawJobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}


// Scraping Sodeico Jobs avec filtrage jobExist
async function scrapeSodeicoJobs(page) {
  const url = "https://sodeico.softy.pro/offres";

  // 1️⃣ Récupération brute des offres
  const rawJobs = await scrapeWithRetry(
    page,
    url,
    "a.card",
    () => {
      const jobList = [];

      document.querySelectorAll("a.card").forEach((job) => {
        // ----- Lien -----
        const relativeLink = job.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://sodeico.softy.pro${relativeLink}`;

        // ----- Titre -----
        const titleEl = job.querySelector("h2.title");
        const title = titleEl?.innerText.trim() || "Non spécifié";

        // ----- Localisation -----
        const locationEl = job.querySelector(".localisation p");
        const location = locationEl?.innerText.trim() || "";

        // ----- Date de publication -----
        const dateEl = job.querySelector("p.time");
        const postedDate = dateEl?.innerText.replace("Mise en ligne le", "").trim() || "";

        // ----- Type de contrat -----
        const contractEl = job.querySelector(".card-tags ul li");
        const contract = contractEl?.innerText.trim() || "";

        // ----- Résumé -----
        const summaryEl = job.querySelector(".card-body p");
        const summary = summaryEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "Sodeico Jobs",
          link,
        });
      });

      return jobList;
    }
  );

  // 2️⃣ Filtrage asynchrone avec jobExist
  const filteredJobs = [];
  for (const job of rawJobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

// Scraping Youth.cd – Offres d'emploi RDC avec filtrage jobExist
async function scrapeYouthCDJobs(page) {
  const url = "https://youth.cd/Opportunites/OffreEmploi";

  // 1️⃣ Récupération brute des offres
  const rawJobs = await scrapeWithRetry(
    page,
    url,
    "article",
    () => {
      const jobList = [];

      document.querySelectorAll("article").forEach((job) => {
        // ----- Titre & lien -----
        const titleEl = job.querySelector("h3 a");
        if (!titleEl) return;

        const title = titleEl.innerText.trim();
        const relativeLink = titleEl.getAttribute("href") || "#";
        const link = relativeLink.startsWith("http")
          ? relativeLink
          : `https://youth.cd${relativeLink}`;

        // ----- Organisation -----
        const orgEl = job.querySelector("p span:not(.job-type)");
        const organization = orgEl?.innerText.trim() || "Confidentiel";

        // ----- Vérification si Expirée -----
        const expiredEl = job.querySelector("span.job-type");
        if (expiredEl && expiredEl.innerText.toLowerCase().includes("expirée")) return;

        // ----- Localisation -----
        const locationEl = job.querySelector(".col-md-3 .brows-job-location p");
        const location = locationEl?.innerText.replace("📍", "").trim() || "";

        // ----- Date de publication -----
        const dateEl = job.querySelector(".col-md-2 .brows-job-location p");
        const postedDate = dateEl?.innerText.trim() || "";

        jobList.push({
          title,
          source: "Youth.cd",
          link,
          
        });
      });

      return jobList;
    }
  );

  // 2️⃣ Filtrage asynchrone avec jobExist
  const filteredJobs = [];
  for (const job of rawJobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs;
}

// Scrape jobrapide.org
async function scrapeJobRapideRDC(page) {

  const rawJobs = await scrapeWithRetry(
    page,
    "https://www.jobrapide.org/?s=&scat=-1&sregion=247&ssecteur=-1&search=offres#google_vignette",
    "div.article",
    () => {
      const jobList = [];
      document.querySelectorAll("div.article").forEach((job) => {
        const title = job.querySelector("h3 a")?.innerText.trim() || "";
        const link = job.querySelector("h3 a")?.getAttribute("href") || "#";
        jobList.push({ title, source: "JobRapide", link });
      });
      return jobList;
    }
  );

  // 2️⃣ Filtrage asynchrone avec jobExist
  const filteredJobs = [];
  for (const job of rawJobs) {
    const nb = await jobExistv2(job.link);
    if (nb === 0) filteredJobs.push(job);
  }

  // 3️⃣ Retourne uniquement les offres valides
  return filteredJobs


}

async function scrapeAfriopp(page) {
  
  const url = "https://afriopp.com/job-location/kinshasa";

  // Naviguer jusqu'à la page et attendre le chargement complet
  await page.goto(url, { waitUntil: "networkidle2", timeout: 0 });

  // Attendre qu'au moins une offre d'emploi apparaisse
  await page.waitForSelector("div.item-job", { timeout: 15000 });

  // Extraire les données directement depuis le DOM rendu
  const jobs = await page.evaluate(() => {
    const jobList = [];

    document.querySelectorAll("div.item-job").forEach((block) => {
      const titleEl = block.querySelector("h2.job-title a");
      const title = titleEl?.innerText.trim() || "";
      const link = titleEl?.href || "";
      const city = block.querySelector(".job-location a")?.innerText.trim() || "";
      const type = block.querySelector(".job-type a")?.innerText.trim() || "";

      if (title && link) {
        jobList.push({
          title,
          source: "Afriopp",
          link,
        });
      }
    });

    return jobList;
  });

  return jobs;
}


//Scraping GoAfrica Online
async function scrapeAfrioppKinshasa(page) {
  const jobs = await scrapeWithRetry(
    page,
    "https://afriopp.com/job-location/kinshasa",
    "div.item-job.col-sm-12.col-md-12.col-xs-12.lg-clearfix.md-clearfix",
    () => {
      const jobList = [];

      document.querySelectorAll("div.item-job.col-sm-12.col-md-12.col-xs-12.lg-clearfix.md-clearfix").forEach((block) => {
        const titleEl = block.querySelector("h2.job-title a");
        const title = titleEl?.innerText.trim() || "";
        const link = titleEl?.href || "";
        const city = block.querySelector("div.job-location a")?.innerText.trim() || "";
        const type = block.querySelector("div.job-type a.type-job")?.innerText.trim() || "";

        if (title && link) {
          jobList.push({
            title,
            source: "Afriopp",
            link
          });
        }
      });

      return jobList;
    }
  );

  // Retourne uniquement les offres valides
  return jobs;
}




//Hiring.Cafe
async function scrapeHiringCafeRDC(page) {
  const jobs = [];
  // Une seule page pour cette recherche précise
  const url =
    "https://hiring.cafe/?searchState=%7B%22locations%22%3A%5B%7B%22id%22%3A%22eRY1yZQBoEtHp_8UEq3V%22%2C%22types%22%3A%5B%22country%22%5D%2C%22address_components%22%3A%5B%7B%22long_name%22%3A%22RDC%22%2C%22short_name%22%3A%22GA%22%2C%22types%22%3A%5B%22country%22%5D%7D%5D%2C%22formatted_address%22%3A%22RDC%22%2C%22population%22%3A2119275%2C%22workplace_types%22%3A%5B%5D%2C%22options%22%3A%7B%22flexible_regions%22%3A%5B%22anywhere_in_continent%22%2C%22anywhere_in_world%22%5D%7D%7D%5D%2C%22dateFetchedPastNDays%22%3A4%7D";

  const pageJobs = await scrapeWithRetry(
    page,
    url,
    "div.relative.xl\\:z-10",
    () => {
      const jobList = [];
      document.querySelectorAll("div.relative.xl\\:z-10").forEach((job) => {
        const title =
          job.querySelector("span.font-bold.text-start")?.innerText.trim() ||
          "Non spécifié";
        const company =
          job.querySelector("span.font-bold")?.innerText
            .replace(":", "")
            .trim() || "Confidentiel";
        const location =
          job
            .querySelector("div.mt-1.flex.items-center span.line-clamp-2")
            ?.innerText.trim() || "Non spécifié";
        const description =
          job
            .querySelector("div.flex.space-x-1 span.line-clamp-6")
            ?.innerText.trim() || "";
        const link =
          job.querySelector('a[href*="company="]')?.href || "#";

        jobList.push({
          title,
          source: "Hiring Cafe",
          link,
          
        });
      });
      return jobList;
    }
  );

  jobs.push(...pageJobs);
  await new Promise((r) => setTimeout(r, REQUEST_INTERVAL));
  return jobs;
}



// Scrape LinkedIn jobs
async function scrapeLinkedinJobs(formattedDate) {
  const jobs = [];
  const API_KEY = "68417f9ad00340d810f2a702";
  const BASE_URL = "https://api.scrapingdog.com/linkedinjobs/";
  const params = {
    api_key: API_KEY,
    field: "*",
    geoid: "103798675", // RDC geoid 
    sort_by: "day",
  };
  for (let page = 1; page <= 5; page++) {
    params.page = page;
    try {
      const response = await axios.get(BASE_URL, { params, timeout: TIMEOUT });
      if (response.status === 200) {
        const data = response.data;
        const formattedJobs = data.map((job) => ({
          title: job.job_position,
          source: "LinkedIn",
          link: job.job_link,
        }));
        jobs.push(...formattedJobs);
        logger.info(`Page ${page} retrieved successfully from LinkedIn`);
        if (data.length < 10) break;
      } else {
        logger.warn(`Request failed for LinkedIn page ${page} with status code: ${response.status}`);
      }
    } catch (error) {
      logger.error("Error scraping LinkedIn jobs:", { error: error.message });
    }
    await new Promise((r) => setTimeout(r, REQUEST_INTERVAL));
  }
  return jobs;
}

//Scrape google jobs
async function scrapeGoogleJobs(maxJobs = 200) {
  const api_key = "68417f9ad00340d810f2a702";
  const url = "https://api.scrapingdog.com/google_jobs/";

  let allJobs = [];
  let nextToken = "";

  const baseParams = {
    api_key,
    query: "offres d'emplois République démocratique du Congo",
    country: "cd",
  };

  let page = 1;

  do {
    console.log(`📄 Récupération de la page ${page}...`);

    const params = { ...baseParams };
    if (nextToken) params.next_page_token = nextToken;

    try {
      const response = await axios.get(url, { params });

      if (response.status === 200 && response.data) {
        const data = response.data;
        const jobs = data.jobs_results || [];

        if (jobs.length === 0) {
          console.log("⚠️ Aucune nouvelle offre trouvée, arrêt.");
          break;
        }

        const processedJobs = jobs.map((job) => ({
          title: job.title || "Non spécifié",
          source: "Google Jobs",
          link:
            job.apply_options?.[0]?.link ||
            job.job_highlights?.[0]?.title ||
            null,
          company: job.company_name || "Non spécifiée",
          location: job.location || "Non précisée",
          postedDate: job.detected_extensions?.posted_at || "",
        }));

        allJobs = allJobs.concat(processedJobs);
        console.log(`✅ Page ${page} : ${processedJobs.length} offres récupérées`);

        // Pagination
        nextToken = data.scrapingdog_pagination?.next_page_token || "";
        page++;

        // Pause entre les requêtes pour éviter le bannissement
        await new Promise((r) => setTimeout(r, 2500));

        // Stopper si trop d’offres
        if (allJobs.length >= maxJobs) break;
      } else {
        console.warn("❌ Erreur HTTP : " + response.status);
        break;
      }
    } catch (error) {
      console.error("🚨 Erreur lors de la récupération :", error.message);
      break;
    }
  } while (nextToken && allJobs.length < maxJobs);

  console.log(`🎯 Total : ${allJobs.length} offres récupérées sur Google Jobs`);
  return allJobs;
}





const safeExecute = async (fn, name, browser, page, maxRetries = 3) => {
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      // Vérifier et réinitialiser la page si nécessaire
      page = await ensurePageIsValid(browser, page);
      
      // Exécuter la fonction de scraping
      const result = await fn();
      
      // Vérifier que le résultat est un tableau valide
      if (!Array.isArray(result)) {
        console.warn(`⚠️ ${name}: Résultat invalide (non tableau), retour d'un tableau vide`);
        return [];
      }

      console.log(`✅ ${name} terminée, ${result.length} jobs récupérés.`);
      return result;
    } catch (error) {
      console.error(`❌ Tentative ${attempt} échouée pour ${name}: ${error.message}`);
      if (attempt === maxRetries) {
        console.error(`❌ Échec définitif pour ${name} après ${maxRetries} tentatives`);
        await sendLogs("RDC", "Erreur", `Échec après ${maxRetries} tentatives pour ${name}: ${error.message}`);
        return [];
      }
      // Attente exponentielle avant la prochaine tentative
      await new Promise(resolve => setTimeout(resolve, 1000 * attempt));
    }
  }
  return []; // Fallback si toutes les tentatives échouent
};

// Fonction pour vérifier et réinitialiser la page si nécessaire
async function ensurePageIsValid(browser, page) {
  try {
    await page.evaluate(() => {}); // Teste si la page est accessible
    return page;
  } catch (error) {
    console.warn("Page invalide, création d'une nouvelle page...");
    if (page) await page.close().catch(() => {});
    return await browser.newPage();
  }
}

// Main scraping function with concurrency control and email notifications
async function scrapeJobs() {
  const pLimitInstance = await pLimit();
  const limit = pLimitInstance(CONCURRENCY_LIMIT);
  let allJobs = [];
  let browser, page;
  const startTime = new Date();

  try {
    ({ browser, page } = await initializeBrowser());

    const scrapingFunctions = [
      { fn: () => scrapeAfriopp(page), name: 'scrapeAfriopp' },
      
      { fn: () => scrapeRadioOkapiRDC(page), name: 'scrapeRadioOkapiRDC' },
      { fn: () => scrapeMonCongoJobs(page), name: 'scrapeMonCongoJobs' },
      { fn: () => scrapeUNJobNetRDC(page), name: 'scrapeUNJobNetRDC' },
      { fn: () => scrapeBensizweJobs(page), name: 'scrapeBensizweJobs' },
      { fn: () => scrapeSodeicoJobs(page), name: 'scrapeSodeicoJobs' },
      { fn: () => scrapeEmploiRDC(page), name: 'scrapeEmploiRDC' },
      { fn: () => scrapeYouthCDJobs(page), name: 'scrapeYouthCDJobs' },
      { fn: () => scrapeJobRapideRDC(page), name: 'scrapeJobRapideRDC' },
      { fn: () => scrapeGoAfricaRDC(page), name: 'scrapeGoAfricaRDC' },
      { fn: () => scrapeDevelopmentAidRDC(page), name: 'scrapeDevelopmentAidRDC' },
      { fn: () => scrapeDevexRDC(page), name: 'scrapeDevexRDC' },
      { fn: () => scrapeMichaelPageRDC(page), name: 'scrapeMichaelPageRDC' },
      { fn: () => scrapeImpactPoolRDC(page), name: 'scrapeImpactPoolRDC' },
      { fn: () => scrapeTalaComRDC(page), name: 'scrapeTalaComRDC' },
      { fn: () => scrapeTonJob(page), name: 'scrapeTonJob' },
      { fn: () => scrapeOptionCarriereRDC(page), name: 'scrapeOptionCarriereRDC' },
      { fn: () => scrapeReliefWebRDC(page), name: 'scrapeReliefWebRDC' },
      { fn: () => scrapeUnjobsRDC(page), name: 'scrapeUnjobsRDC' },
      { fn: () => scrapeHiringCafeRDC(page), name: 'scrapeHiringCafeRDC' },
      { fn: () => scrapeLinkedinJobs(new Date().toISOString().split("T")[0]), name: 'scrapeLinkedinJobs' },
      { fn: () => scrapeGoogleJobs(page), name: 'scrapeGoogleJobs' },
      
    ];

    //const jobsResults = await Promise.all(scrapingFunctions.map((fn) => limit(() => fn())));
    
    const jobsResults = [];

    let successCount=0;
    let errorCount=0;

    for (const { fn, name } of scrapingFunctions) {
    try {
      const jobs = await limit(() => safeExecute(fn, name, browser, page));
      if (jobs && jobs.length > 0) {
        successCount++;
        jobsResults.push(...jobs);
      } else {
        errorCount++;
        await sendLogs("RDC", "Erreur", `Erreur d'extraction sur ${name}`);
      }
    } catch (error) {
      errorCount++;
      await sendLogs("RDC", "Erreur", `Exception inattendue sur ${name}: ${error.message}`);
    }
  }

  console.log(`✅ ${successCount} réussis, ❌ ${errorCount} échoués`);
  console.log(`Nombre total d'offres: ${jobsResults.length}`);



    allJobs = jobsResults.flat();
    logger.info(`Total jobs collected: ${allJobs.length}`);

    for (const [index, job] of allJobs.entries()) {
      await new Promise((r) => setTimeout(r, REQUEST_INTERVAL));
      try {
        const exists = await jobExist(job.link);
        if (exists == 0) {
          

          const liste_specialites = await getListeSpecialites();
          
          const [specialite, lettre_motivation, indications_cv, langue] = await Promise.all([
          detecterSpecialite(job.title,liste_specialites).catch(() => -1),
          getLettreMotivation(job.title).catch(() => -1),
          generateCVInstructions(job.title).catch(() => -1),
          detectJobTitleLanguage(job.title).catch(() => -1),
          ]);
          
          const liste_profils = await getListeProfils(specialite);
          const profil = await detecterSpecialite(job.title, liste_profils);
          
          console.log(specialite+" "+profil);

          const type = job.source.toLowerCase().includes("remote") || job.source.toLowerCase().includes("nomads")
            ? "Télétravail"
            : "En présentiel";
          if (specialite !== -1 && lettre_motivation !== -1 && indications_cv !== -1 && langue !== -1) {
            await sendJobOffer(
              job.title,
              job.link,
              job.source,
              "RDC",
              specialite,
              profil,
              type,
              lettre_motivation,
              indications_cv,
              langue
            );
            logger.info(`Job sent: ${job.title}`);
          } else {
            logger.warn(`Invalid data for job "${job.title}", skipped`);
          }
        } else {
          logger.info(`Job already exists: ${job.title}`);
        }

        
      } catch (error) {
        logger.error(`Error processing job "${job.title}":`, { error: error.message });
      }
    }

    
    await sendLogs("RDC", "Succès", `${allJobs.length} Offres enregistrées avec succès`);


  } catch (error) {
    logger.error("Error in scrapeJobs:", { error: error.message });
    // Send error email
    await sendLogs("RDC", "Erreur", `Échec du scraping des offres d'emploi`);

    throw error;
  } finally {
    if (browser) await browser.close();
    logger.info("Browser closed");
  }
}

async function getListeSpecialites() {
  try {
    const response = await axios.get("https://job-finder-bot.com/home/liste_specialites");
    
    if (response.data.status === "success") {
      //console.log("Liste des spécialités :", response.data.specialites);
      return response.data.specialites;
    } else {
      console.log("Erreur : statut non valide");
      return [];
    }
  } catch (error) {
    console.error("Erreur lors de la récupération :", error.message);
    return [];
  }
}


async function getListeProfils(specialite) {
  try {
    const response = await axios.post(
      'https://job-finder-bot.com/home/liste_profils', // 🔹 URL de ton endpoint PHP
      {
        specialite: specialite, // 🔹 Donnée envoyée au backend PHP
      },
      {
        headers: {
          'Content-Type': 'application/json',
        },
      }
    );

    if (response.data.status === 'success') {
      //console.log('✅ Profils récupérés :', response.data.profils);
      return response.data.profils;
    } else {
      console.error('❌ Erreur côté PHP :', response.data);
      return null;
    }

  } catch (error) {
    console.error('🚨 Erreur Axios :', error.message);
    return null;
  }
}

function detectionLocale(jobTitle, listeSpecialites) {
  const titre = jobTitle.toLowerCase().normalize("NFD").replace(/[\u0300-\u036f]/g, "");

  for (const specialite of listeSpecialites) {
    const spec = specialite.toLowerCase().normalize("NFD").replace(/[\u0300-\u036f]/g, "");
    // Vérifie si le mot de la spécialité apparaît dans le titre
    if (titre.includes(spec)) {
      return specialite; // correspondance trouvée
    }
  }

  return null; // aucune correspondance locale
}

/**
 * Détecte la spécialité la plus pertinente via l'API OpenRouter
 */
async function detecterSpecialite(jobTitle, listeSpecialites, maxRetries = 3, delayMs = 2000) {
  const apiKey = API_KEY;
  if (!apiKey) {
    throw new Error("Clé API OpenRouter manquante. Définis OPENROUTER_API_KEY dans ton fichier .env");
  }

  // Nettoyage de la liste
  const specialitesArray = listeSpecialites
    .split(",")
    .map(s => s.trim())
    .filter(Boolean);

  // 🔍 Étape 1 : détection locale rapide
  const locale = detectionLocale(jobTitle, specialitesArray);
  if (locale) {
    console.log(`✅ Détection locale : "${locale}" trouvée dans le titre "${jobTitle}"`);
    return locale;
  }

  // 🧠 Étape 2 : si aucune correspondance locale → appel à l'API Claude
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      const response = await axios.post(
        "https://openrouter.ai/api/v1/chat/completions",
        {
          model: "anthropic/claude-3.5-sonnet",
          messages: [
            {
              role: "system",
              content: `Tu es un assistant qui classe les offres d’emploi selon leur spécialité.
              Réponds toujours uniquement dans ce format JSON :
              {"specialite": "nom_de_la_spécialité"}
              La valeur "nom_de_la_spécialité" doit être une des spécialités suivantes :
              ${specialitesArray.join(", ")}
              Si aucune ne correspond clairement, réponds exactement :
              {"specialite": "Aucune correspondance"}.`
            },
            {
              role: "user",
              content: `Titre de l'offre : "${jobTitle}".
              Donne la spécialité la plus pertinente correspondant à ce titre.`
            }
          ],
          temperature: 0.2,
          max_tokens: 50
        },
        {
          headers: {
            "Authorization": `Bearer ${apiKey}`,
            "Content-Type": "application/json"
          },
          timeout: 15000
        }
      );

      const raw = response.data.choices[0].message.content.trim();
      let resultat = "Aucune correspondance";

      try {
        const parsed = JSON.parse(raw);
        if (parsed && parsed.specialite) resultat = parsed.specialite.trim();
      } catch {
        const match = raw.match(/"([^"]+)"/);
        resultat = match ? match[1].trim() : raw.replace(/[^a-zA-ZÀ-ÿ0-9 \-]/g, "").trim();
      }

      // Vérifie que la spécialité existe dans la liste
      const correspondance = specialitesArray.find(
        s => s.toLowerCase() === resultat.toLowerCase()
      );

      if (!correspondance) {
        return "Aucune correspondance";
      }

      return correspondance;

    } catch (error) {
      console.warn(`❌ Tentative ${attempt} échouée:`, error.message);
      if (attempt === maxRetries) {
        console.error("Toutes les tentatives ont échoué. Retourne 'Aucune correspondance'.");
        return "Aucune correspondance";
      }

      const waitTime = delayMs * attempt;
      console.log(`🔁 Nouvelle tentative dans ${waitTime / 1000} secondes...`);
      await new Promise(res => setTimeout(res, waitTime));
    }
  }
}


async function getListeSpecialites() {
  try {
    const response = await axios.get("https://job-finder-bot.com/home/liste_specialites");
    
    if (response.data.status === "success") {
      //console.log("Liste des spécialités :", response.data.specialites);
      return response.data.specialites;
    } else {
      console.log("Erreur : statut non valide");
      return [];
    }
  } catch (error) {
    console.error("Erreur lors de la récupération :", error.message);
    return [];
  }
}


async function getListeProfils(specialite) {
  try {
    const response = await axios.post(
      'https://job-finder-bot.com/home/liste_profils', // 🔹 URL de ton endpoint PHP
      {
        specialite: specialite, // 🔹 Donnée envoyée au backend PHP
      },
      {
        headers: {
          'Content-Type': 'application/json',
        },
      }
    );

    if (response.data.status === 'success') {
      //console.log('✅ Profils récupérés :', response.data.profils);
      return response.data.profils;
    } else {
      console.error('❌ Erreur côté PHP :', response.data);
      return null;
    }

  } catch (error) {
    console.error('🚨 Erreur Axios :', error.message);
    return null;
  }
}

function detectionLocale(jobTitle, listeSpecialites) {
  const titre = jobTitle.toLowerCase().normalize("NFD").replace(/[\u0300-\u036f]/g, "");

  for (const specialite of listeSpecialites) {
    const spec = specialite.toLowerCase().normalize("NFD").replace(/[\u0300-\u036f]/g, "");
    // Vérifie si le mot de la spécialité apparaît dans le titre
    if (titre.includes(spec)) {
      return specialite; // correspondance trouvée
    }
  }

  return null; // aucune correspondance locale
}

/**
 * Détecte la spécialité la plus pertinente via l'API OpenRouter
 */
async function detecterSpecialite(jobTitle, listeSpecialites, maxRetries = 3, delayMs = 2000) {
  const apiKey = API_KEY;
  if (!apiKey) {
    throw new Error("Clé API OpenRouter manquante. Définis OPENROUTER_API_KEY dans ton fichier .env");
  }

  // Nettoyage de la liste
  const specialitesArray = listeSpecialites
    .split(",")
    .map(s => s.trim())
    .filter(Boolean);

  // 🔍 Étape 1 : détection locale rapide
  const locale = detectionLocale(jobTitle, specialitesArray);
  if (locale) {
    console.log(`✅ Détection locale : "${locale}" trouvée dans le titre "${jobTitle}"`);
    return locale;
  }

  // 🧠 Étape 2 : si aucune correspondance locale → appel à l'API Claude
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      const response = await axios.post(
        "https://openrouter.ai/api/v1/chat/completions",
        {
          model: "anthropic/claude-3.5-sonnet",
          messages: [
            {
              role: "system",
              content: `Tu es un assistant qui classe les offres d’emploi selon leur spécialité.
              Réponds toujours uniquement dans ce format JSON :
              {"specialite": "nom_de_la_spécialité"}
              La valeur "nom_de_la_spécialité" doit être une des spécialités suivantes :
              ${specialitesArray.join(", ")}
              Si aucune ne correspond clairement, réponds exactement :
              {"specialite": "Aucune correspondance"}.`
            },
            {
              role: "user",
              content: `Titre de l'offre : "${jobTitle}".
              Donne la spécialité la plus pertinente correspondant à ce titre.`
            }
          ],
          temperature: 0.2,
          max_tokens: 50
        },
        {
          headers: {
            "Authorization": `Bearer ${apiKey}`,
            "Content-Type": "application/json"
          },
          timeout: 15000
        }
      );

      const raw = response.data.choices[0].message.content.trim();
      let resultat = "Aucune correspondance";

      try {
        const parsed = JSON.parse(raw);
        if (parsed && parsed.specialite) resultat = parsed.specialite.trim();
      } catch {
        const match = raw.match(/"([^"]+)"/);
        resultat = match ? match[1].trim() : raw.replace(/[^a-zA-ZÀ-ÿ0-9 \-]/g, "").trim();
      }

      // Vérifie que la spécialité existe dans la liste
      const correspondance = specialitesArray.find(
        s => s.toLowerCase() === resultat.toLowerCase()
      );

      if (!correspondance) {
        return "Aucune correspondance";
      }

      return correspondance;

    } catch (error) {
      console.warn(`❌ Tentative ${attempt} échouée:`, error.message);
      if (attempt === maxRetries) {
        console.error("Toutes les tentatives ont échoué. Retourne 'Aucune correspondance'.");
        return "Aucune correspondance";
      }

      const waitTime = delayMs * attempt;
      console.log(`🔁 Nouvelle tentative dans ${waitTime / 1000} secondes...`);
      await new Promise(res => setTimeout(res, waitTime));
    }
  }
}




async function getLettreMotivation(jobTitle) {
  const headers = {
    'Authorization': `Bearer ${API_KEY}`,
    'Content-Type': 'application/json'
  };

  const systemMessage = {
    role: "system",
    content: `
      You are an expert in writing cover letters.
      For the following job title: "${jobTitle}", detect the language of the job title (e.g., English or French).
      Write a professional and concise cover letter in the same language as the job title.
      - Use a formal tone appropriate for the role.
      - Mention relevant skills and experiences.
      - Do not include personal information.
      - Structure: introduction, body, conclusion.
      - Return only the body content of the cover letter, excluding any subject line, email header ( "Objet : Candidature").
      - Use plain text with line breaks between paragraphs.
    `
  };

  const userMessage = {
    role: "user",
    content: `Write only the body content of a cover letter for the position: "${jobTitle}" in the same language as the job title, excluding any subject line, email header, or salutations.`
  };

  const data = {
    model: "mistral/ministral-8b",
    messages: [systemMessage, userMessage],
    temperature: 0.5
  };

  try {
    const response = await axios.post(API_URL, data, { headers });
    const choices = response?.data?.choices;
    if (!choices || !choices[0]?.message?.content) {
      return -1;
    }
    const rawLetter = choices[0].message.content.trim();
    const htmlContent = `
      <div class="max-w-2xl mx-auto p-6 bg-white rounded-lg shadow-md">
        <div class="text-gray-700 space-y-4">
          ${rawLetter.split('\n\n').slice(0).map(paragraph => `<p>${paragraph}</p>`).join('')}
        </div>
        <div class="mt-6 text-gray-700">
          <p>@prenom@ @nom@</p>
        </div>
      </div>
    `;
    return htmlContent;
  } catch (error) {
    logger.error("Erreur de génération de la lettre:", { error: error.message });
    return -1;
  }
}

async function generateCVInstructions(jobTitle) {
  const prompt = `
Vous êtes un expert en recrutement avec une expertise approfondie en optimisation de CV. Pour une offre d'emploi intitulée "${jobTitle}", détectez la langue du titre (français ou anglais) et rédigez des instructions détaillées dans la même langue pour optimiser un CV afin d'augmenter les chances d'être retenu. Adaptez les conseils au poste spécifique indiqué dans le titre.

Structurez les instructions en trois sections :
- **Présentation du candidat** (ou **Candidate Presentation** en anglais),
- **Compétences** (ou **Skills** en anglais),
- **Expériences professionnelles** (ou **Professional Experience** en anglais).

Pour chaque section, incluez deux sous-sections avec les balises <h3> :
- **À inclure** : Détaillez précisément ce qu’il faut inclure, avec des exemples concrets adaptés au poste.
- **À éviter** : Indiquez ce qu’il ne faut pas inclure, avec des explications claires sur pourquoi ces éléments peuvent nuire.

### Détails par section :
1. **Présentation du candidat**/**Candidate Presentation** :
   - **À inclure** : Un résumé concis (3-4 phrases) des qualifications clés, de la motivation pour le poste, et de l’adéquation avec les besoins de l’employeur. Mentionnez des compétences ou expériences clés tirées du titre du poste.
   - **À éviter** : Les déclarations génériques, les informations personnelles (âge, état civil), ou les objectifs trop vagues.
2. **Compétences**/**Skills** :
   - **À inclure** : Une liste de compétences techniques et interpersonnelles pertinentes pour le poste, priorisées selon le titre (ex. : langages de programmation pour un poste de développeur, compétences en gestion pour un poste de manager). Indiquez comment les présenter (ex. : bullet points, niveau de maîtrise si pertinent).
   - **À éviter** : Les compétences obsolètes, non pertinentes pour le poste, ou listées sans contexte.
3. **Expériences professionnelles**/**Professional Experience** :
   - **À inclure** :
     - Les postes précédents les plus pertinents pour le titre du poste.
     - Les missions ou projets marquants, avec des descriptions spécifiques.
     - Les responsabilités principales tenues.
     - Les résultats mesurables obtenus (ex. : augmentation de 20 % des ventes, livraison d’un projet dans les délais).
     - Les technologies, outils ou méthodologies utilisés, si pertinents pour le poste.
   - **À éviter** : Les expériences non pertinentes, les descriptions trop vagues, ou l’absence de résultats quantifiables.

Retournez EXCLUSIVEMENT le code HTML structuré à insérer directement dans la balise <body> d'une page web, avec des balises <h2> pour les sections principales, <h3> pour les sous-sections ("À inclure", "À éviter"), et <p> ou <ul><li> pour les détails. N'incluez AUCUNE balise Markdown, AUCUNE balise <html>, <head>, ou <body>, AUCUN texte explicatif, AUCUN commentaire, et AUCUN contenu non-HTML.
Si vous ne pouvez pas déterminer la langue, utilisez le français par défaut.
`;

  try {
    const response = await axios.post(API_URL, {
      model: 'mistral/ministral-8b',
      messages: [
        { role: 'system', content: 'Vous êtes un assistant spécialisé en rédaction de CV et capable de détecter la langue d’un titre d’offre d’emploi.' },
        { role: 'user', content: prompt }
      ]
    }, {
      headers: {
        'Authorization': `Bearer ${API_KEY}`,
        'Content-Type': 'application/json'
      }
    });
    let htmlContent = response.data.choices[0].message.content;
    htmlContent = htmlContent
      .replace(/```(?:html|[a-z]*)?[\r\n]?/gi, '')
      .replace(/```/g, '')
      .replace(/<\s*body[^>]*>/gi, '')
      .replace(/<\s*\/\s*body\s*>/gi, '')
      .trim();
    if (!htmlContent.includes('<h2>') || !htmlContent.match(/<[^>]+>/)) {
      logger.error('Erreur : La réponse ne contient pas de HTML valide après nettoyage.');
      return -1;
    }
    return htmlContent;
  } catch (error) {
    logger.error('Erreur lors de la requête à l\'API OpenRouter:', { error: error.message });
    return -1;
  }
}

async function detectJobTitleLanguage(jobTitle) {
  const prompt = `
Vous êtes un expert en linguistique. Analysez le titre d'offre d'emploi suivant : "${jobTitle}". Déterminez si le titre est en français ou en anglais. Retournez uniquement "fr" si le titre est en français, "en" si le titre est en anglais. Si vous ne pouvez pas déterminer la langue avec certitude, retournez "fr" par défaut.
  `;
  try {
    const response = await axios.post(API_URL, {
      model: 'mistral/ministral-8b',
      messages: [
        { role: 'system', content: 'Vous êtes un assistant spécialisé en détection de langue.' },
        { role: 'user', content: prompt }
      ]
    }, {
      headers: {
        'Authorization': `Bearer ${API_KEY}`,
        'Content-Type': 'application/json'
      }
    });
    const language = response.data.choices[0].message.content.trim();
    return language === 'fr' || language === 'en' ? language : 'fr';
  } catch (error) {
    logger.error('Erreur lors de la requête à l\'API OpenRouter:', { error: error.message });
    return -1;
  }
}

async function sendJobOffer(titre, lien, source, pays, specialite, profil,type, lettre_motivation, indications_cv, langue) {
  const payload = { title: titre, source, type, link: lien, pays, specialite,profil, lettre_motivation, indications_cv, langue };
  try {
    const response = await axios.post("https://job-finder-bot.com/home/ajout_offres", payload, {
      headers: {
        'Content-Type': 'application/json'
      }
    });
    logger.info("Résultat de l'API:", { message: response.data.message });
    return response;
  } catch (error) {
    if (error.response) {
      logger.error("Erreur API:", {
        status: error.response.status,
        data: error.response.data,
        message: error.message
      });
    } else {
      logger.error("Erreur lors de l'envoi à l'API SendJobOffer:", { error: error.message });
    }
    throw error;
  }
}

async function jobExist(link) {
  const url = 'https://job-finder-bot.com/home/testLink';
  const data = { link };
  try {
    const response = await axios.post(url, data);
    logger.info(`Job existence check: ${response.data.nb}`);
    return response.data.nb;
  } catch (error) {
    logger.error('Erreur lors de la requête POST JobExist:', { error: error.message });
    return -1;
  }
}

async function jobExistv2(link) {
  const url = 'https://job-finder-bot.com/home/testLink';
  const data = { link };
  try {
    const response = await axios.post(url, data);
    //logger.info(`Job existence check: ${response.data.nb}`);
    return response.data.nb;
  } catch (error) {
    //logger.error('Erreur lors de la requête POST JobExist:', { error: error.message });
    return -1;
  }
}

schedule.scheduleJob(
  { hour: 10, minute: 0, tz: "Etc/GMT" },  // ou "Etc/UTC"
  async () => {
    logger.info("Scheduled script execution at 06:00 GMT...");
    try {
      await scrapeJobs();
    } catch (err) {
      logger.error("Error during scheduled execution:", { error: err });
      await sendLogs("RDC", "Erreur", `Échec du scraping des offres d'emploi`);
    }
  }
);


// Run immediately for testing
logger.info("Starting initial script execution...");
scrapeJobs().catch((err) => {
  logger.error("Error durant execution RDC:", { error: err });
  
});
